;;;*******************************************************************
;;; NAME:	write.asm
;;;
;;; DESCR:	Implements sector writing.
;;;
;;; NOTES:	
;;;*******************************************************************

#include "vd.h"
#include "write.mac"
#include "writeMFM.mac"
#include "nop.mac"

#define ShiftIn		rlf
#define ShiftOut	rrf
	
;;;
;;; THISBIT - set this bit and it will be rotated into the current byte
;;; LASTBIT - used to access the last bit that was rotated in
;;; NEXTBIT - set this to cause "THISBIT" to be set after a rotate
;;;
;;; (NOTE - if "ShiftIn" changes to rrf, then LASTBIT and NEXTBIT need
;;;  to reverse.)
	
#define THISBIT		STATUS,C
#define LASTBIT		MEMDATA,0
#define NEXTBIT		MEMDATA,7

PROG1 CODE

	global	DDWriteSector
	global	SDWriteSector
	global	H17WriteSector
	
;;;*******************************************************************
;;; NAME:	DDWriteSector()
;;;
;;; DESCR:	Called when a WRITE_GATE assertion is noticed.
;;;		Don't quite know when this is going to get called,
;;;		relative to the actual time that the WRITE_GATE is
;;;		asserted.  So this routine does the following:	
;;;
;;;		- wait for the first pulse
;;;		- now we are "in sync" with the string of zeros that
;;;		  are coming on the write line.  Each separated by
;;;		  20 cycles (4000 ns).
;;;		- the moment we DON'T see a pulse at 20, that means
;;;		  we are seeing the start of the 0xa1 sync bytes.
;;;		- at this point, wait for 15 more pulses (3 * 5)
;;;		- immediately shift to reading the first byte (data mark)
;;;		
;;;
;;; ARGS:	uses WORK0, WORK1, WORK2, WORK3
;;;
;;; RETURNS:
;;;
;;; NOTES:	- the DDSector routine has ALREADY moved the MemAddr
;;;		  to the data portion by the time this routine is
;;;		  called...so this routine moves it right back
;;;		- the MemIncr was NOT done, so memory is pointing
;;;		  at the data mark as expected
;;;*******************************************************************
DDWriteSector:

	movlw	'W'		; DEBUG
	movwf	TXREG		;   don't wait to clear it...let it go...
	
	;; first, wait for a pulse, any pulse...we then get "in sync."
	;; Note that due to this loop, which has 3 cycles between samples,
	;; we CAN miss a pulse, but only one...the next one would be caught
	;; because there will be 20 cycles between them.

	SkipIfHigh	WDATA		; (0)loop here waiting for a pulse
	GotoPrevious			; (1)

	;; at this point, we're "in" a pulse.  For now, we assume that
	;; we caught the pulse at its beginning, labeling that time zero (0).
	;; We know, however, that we actually could be anywhere in the pulse,
	;; meaning that it may have actually started between -500ns and 0ns
	;; from our "zero" time (pulses are 500ns wide).

	;; That range, though, is too big.  We need to narrow it down.
	;; We do that by sampling between the 500ns range until we
	;; get it bounded.

	Nop16				; (2) wait for the next pulse
	SkipIfLow	WDATA		; (18) sample at 18 (-400ns)
	goto		DDWS_18		; (19) we found the pulse at 18

	;; need to let a pulse pass, waiting for the next one
	
	Nop15				; (0) wait for the next pulse
	Nop4				; (15)
	SkipIfLow	WDATA		; (19) sample at 19 (-200ns)
	goto		DDWS_19		; (0) we found the pulse at 19

	goto		DDWS_20		; (1) otherwise, pulse must be at 20

DDWS_18:				; (1 --> 3)
	;; at this point we have narrowed the pulse down to starting
	;; between -500 and -400ns from the point we noticed it.  So
	;; we adjust our concept of "zero" accordingly.  This gets us
	;; to a concept of "zero" where the pulse could have actually
	;; started between -100ns and 0ns prior.  This adjustment is
	;; done by adding 2 pulses to our concept of where were thought
	;; we were.

DDWS_19:				; (2 --> 3)
	;; at this point we have narrowed the pulse down to starting
	;; between -400 and -200ns from the point we noticed it.  So
	;; we adjust our concept of "zero" accordingly.  This gets us
	;; to a concept of "zero" where the pulse could have actually
	;; started between -200ns and 0ns prior.  This adjustment is
	;; done by adding 1 pulse to our concept of where were thought
	;; we were.

DDWS_20:				; (3 --> 3)
	;; if we got this far, that means that our original sample
	;; was only off by -200ns to 0ns, which is our target range.
	;; So no adjustment to our zero concept is necessary.

	;; NOTE - the above labels just make the code easier to
	;; understand.  There is NOT meant to be any additional code
	;; for DDWS_18, DDWS_19, or DDWS_20.

	;; NOTE - we have "consumed" up to 5 pulses by this time.
	
	Nop2				; (3)
	
DDWS_Loop:

	;; now that we are in-sync, continue monitoring pulses until
	;; we get one that comes at 30 instead of 20...

	;; since we would be just otherwise wasting time here waiting
	;; until we get to 20, we go ahead and set-up a few variables
	;; sure, we set them up over and over again...

	movlw	d'1'			; (5)
	movwf	WORK1			; (6) WORK1 has data mark byte count
	clrf	WORK2			; (7) WORK2 has sector byte count (zero = 256)
	movlw	d'2'			; (8)
	movwf	WORK3			; (9) WORK3 has CRC byte count

	;; IMPORTANT NOTE - sampling occurs at 20 and 22 now that we are
	;; "in sync" with the incoming data pulses.  This dual sampling
	;; is necessary to capture potential precompensation-adjusted
	;; pulses.  See the note on prcompensation in this directory.

	Nop10				; (10)

	SkipIfLow	WDATA		; (20)
	SkipAlways			; (21)
	SkipIfLow	WDATA		; (22)
	goto		DDWS_Loop	; (23,3) found a pulse at 20, keep monitoring
	
	;; at this point, we have begun the sync bytes ('cause we
	;; we didn't get a pulse at 20).

	;; we have a special sync byte coming in, and just to
	;; be "easy" about it...just count the pulses until
	;; we get 15, indicating that we are just about to start
	;; the real data.

	movlw	d'15'			; (24)
	movwf	WORK0			; (25)
	goto	DDWS_Sample2		; (26) jump into the code below
	
	
DDWS_Sample:	
	;; this sequence of code is good enough to capture pulses
	;; that occur at 20, 30, or 40

	Nop12				; (8)
	
	SkipIfLow	WDATA		; (20)
	SkipAlways			; (21)
	SkipIfLow	WDATA		; (22)
	goto		DDWS_Count	; (23) got a pulse, count it

	Nop4				; (24) no pulse....move on
	
DDWS_Sample2:	;; the first sample jumps in here from above
	
	Nop2				; (28)
	
	SkipIfLow	WDATA		; (30)
	SkipAlways			; (31)
	SkipIfLow	WDATA		; (32)
	goto		DDWS_Count	; (33)

	Nop6				; (34)
	
	SkipIfLow	WDATA		; (40)
	SkipAlways			; (41)
	SkipIfLow	WDATA		; (42)
	goto		DDWS_Count	; (43)

	;; falling through here means something went awry
	;; just pretend we got a pulse so we don't loop
	;; forever...
	
DDWS_Count:	;; got a pulse here, count it and re-loop or proceed

	decfsz	WORK0			; (5)
	goto	DDWS_Sample		; (6)

	;; we have just completed 15 pulses, indicating that we are
	;; headed for the first real byte of data (the data mark)

	;; the last bit was a one, so set it appropriately

	bsf	LASTBIT			; (7) sets the LASTBIT as 1
	bsf	WORK0,3			; (8) bitcount set to eight (WORK0 was zero from above)

	;; get back to the base memory location before starting - DDSector had
	;; pre-emptively set it to the data portion even though we're still pointing
	;; at the sector data mark

	MemAddrLoSet			; (9)
	
	;; prepare memory for writing

	MemWriteMode			; (11)

	Nop3				; (15)

	;; note that SECCOUNT has already been incremented since writing
	;; is noticed during the pre-data gap...it doesn't need to be done here
	
	MFM_Write  Wmark,WORK0,WORK1	; (18) write data mark

 	MemAddrInc	SECCOUNT	; (16) flip to data block

	MFM_Write  Wdata,WORK0,WORK2	; (18) write data

	MemAddrLoSet			; (16) back to header block

	MFM_Write  Wcrc,WORK0,WORK3	; (18) write CRC
	
	MemReadMode			; (16)

	;; remember that there is actually 1 more byte of 0xff's coming
	;; out from the WRITE at this point...actually, one of those bits
	;; may have been already read...

	return				; (21)

;;;*******************************************************************
;;; NAME:	SDWriteSector()
;;;
;;; DESCR:	Called when a WRITE_GATE assertion is noticed.
;;;		Much like with DDWriteSector, we don't know when
;;;		this is going to get called relative to the time that
;;;		the WRITE_GATE is asserted, so we do the following:
;;;
;;;		- wait for the first pulse, which is in the middle of
;;;		  the 6 zeros (48 pulses) before the data mark
;;;		- now we are "in sync" with the string of zeros that
;;;		  are coming on the write line.  Each separated by
;;;		  40 cycles (8000 ns).
;;;		- the moment we see a pulse at 20, that means
;;;		  we are seeing the start of the data mark, which is
;;;		  always an 0xfX, where X is an a or b (though it doesn't
;;;		  really matter, as long as it is an f in front)
;;;		- at this point we start loading the data mark, then data & CRC
;;;
;;; ARGS:	uses WORK0, WORK1
;;;
;;; RETURNS:
;;;
;;; NOTES:
;;;*******************************************************************
SDWriteSector:

	movlw	'w'		; DEBUG
	movwf	TXREG		;   don't wait to clear it...let it go...
	
	call	FM_ZeroSync		; (?) sync on the incoming zeros, then the first one bit

	;; at this point we have already seen the first bit of
	;; the data mark
	
	Nop5				; (23)

	bsf	LASTBIT			; (28) we have seen the first one, so LASTBIT = 1
	movlw	7			; (29)
	movwf	WORK0			; (30) set bit count to 7 since we already have one

	;; prepare memory for writing

	MemWriteMode			; (31)

	movlw	1			; (35)
	movwf	WORK1			; (36) number of bytes for data mark
	
	call	FM_Write		; (37) get the rest of the data mark

	;; note that SECCOUNT has already been incremented since writing
	;; is noticed during the pre-data gap...it doesn't need to be done here
	
	MemAddrInc	SECCOUNT	; (33) flip to data block

	clrf	WORK1			; (35) number of bytes for data (256)
	Nop				; (36)
	
	call	FM_Write		; (37) get the data

	MemAddrLoSet			; (33) flip back to header block for CRC
	
	movlw	2			; (35)
	movwf	WORK1			; (36) number of bytes for crc
	
	call	FM_Write		; (37) get the CRC

	MemReadMode			; (33)

	;; remember that there is actually 1 more byte of 0xff's coming
	;; out from the WRITE at this point...

	return				; done!


;;;*******************************************************************
;;; NAME:	H17WriteSector()
;;;
;;; DESCR:	Called when a WRITE_GATE assertion is noticed.
;;;		Much like with SD/DDWriteSector, we don't know when
;;;		this is going to get called relative to the time that
;;;		the WRITE_GATE is asserted, so we do the following:
;;;
;;;		- wait for the first pulse, which is in the middle of
;;;		  the stream of 12 zeros before the sync byte (note
;;;		  that we read the sync back backwards from what it
;;;		  really is.
;;;		- now we are "in sync" with the string of zeros that
;;;		  are coming on the write line.  Each separated by
;;;		  40 cycles (8000 ns).
;;;		- the moment we see a pulse at 20, that means
;;;		  we are seeing the start of the sync byte, which is an
;;;		  0xfd (which is 1111 1101) or backwards 0xbf.
;;;		- at this point we start loading the sync byte, then data & CRC
;;;
;;; ARGS:	uses WORK0, WORK1
;;;
;;; RETURNS:
;;;
;;; NOTES:	- this routine tries to look like the read routine,
;;;		  keeping the index pulses where they belong.
;;;*******************************************************************

;; the h17 sync byte is 0xfd, but it is sent out LSb first
;; 
;; #define REAL_H17_SYNC_BYTE	0xfd

#define H17_SYNC_BYTE	0xbf

H17WriteSector:

	movlw	'h'		; DEBUG
	movwf	TXREG		;   don't wait to clear it...let it go...

	call	FM_ZeroSync		; (?) sync on the incoming zeros, then the first one bit

	;; note that SECCOUNT has already been incremented since writing
	;; is noticed during the pre-data gap...it doesn't need to be done here
	
	MemAddrInc	SECCOUNT	; (23) flip to data block
	MemWriteMode			; (25) and turn on write mode

	;; the sync byte isn't written to memory, it is simply a static piece of
	;; the format for H17.  So we need to continue reading it, keeping in
	;; sync with the rest of the format.
	;; The sync byte (reversed) is 0xbf or 1011 1111, but since the first
	;; bit has already been received, we have 011 1111 to read in, which
	;; is 13 cycles (1 clock for the zero, and 1 clock + 1 data for 6 ones)
	;; So we'll just loop reading in 13 pulses, knowing that we exit the
	;; last one at 23 since it is a one

	Nop6				; (29)
	movlw	d'13'			; (35)
	movwf	WORK0			; (36)
	call	FM_PulseCount		; (37) now count off pulses, staying in sync
	
	;; now prepare to read in data.  The last pulse was a one, so we are on 28.

	bsf	WORK0,3			; (28) WORK0 is zero prior, set it to 8
	bsf	LASTBIT			; (29) LASTBIT = 1 (doesn't really matter to FM)
	
	;; we are generating index pulses here too, remember, so we do the writing as:
	;;	32 bytes, then lower the track index pulse
	;;      96 bytes, then raise the track index pulse if necessary
	;;	64 bytes, then lower the track index pulse in any case
	;;	64 bytes at the end
	;; 

	movlw	d'32'			; (30)
	movwf	WORK1			; (31)
	Nop5				; (32)
	
	call	FM_Write		; (37) get the data

	Lower	INDEX_HOLE		; (33) lower the sector indicator index pulse
	movlw	d'96'			; (34)
	movwf	WORK1			; (35)
	movlw	d'10'			; (36) FM_WRITE doesn't use or modify W
	
	call	FM_Write		; (37) next chunk

	subwf	SECCOUNT,W		; (33) W is d'10' from above - check for track index
	skpnz				; (34)
	Raise	INDEX_HOLE		; (35)
	bsf	WORK1,6			; (36) WORK1 was 0, set bit 6 ==> 64

	call	FM_Write		; (37) next chunk
	
	Lower	INDEX_HOLE		; (33) lower it in any case
	bsf	WORK1,6			; (34) WORK1 was 0, set bit 6 ==> 64
	Nop2				; (35)

	call	FM_Write		; (37) last big chunk

	MemAddrLoSet			; (33) flip back to header block for CRC
	
	movlw	d'1'			; (35)
	movwf	WORK1			; (36) one byte for crc
	
	call	FM_Write		; (37) get the CRC

	MemReadMode			; (33)

	return				; done!

;;;**********************************************************************
;;; NAME:	FM_PulseCount()
;;;
;;; DESCR:	Counts off WORK0 number of FM-style pulses.  Stays in sync
;;;		regardless of the bits.
;;;
;;; ARGS:	WORK0 has the number of pulses to count
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- since it is FM, the pulses are assumed to be at least
;;;		  20 cycles apart.
;;;		- assumes syncronization and calling at -1 (do "call" at -3)
;;;		- returns at 6 (next inst at 8)
;;;**********************************************************************
FM_PulseCount:
	WriteSample3	FM_PCNext	; (-1)

	goto		FM_PCNext2	; (5)

FM_PCNext:
	Nop1				; (3)
	decfsz		WORK0		; (4)
	goto		FM_PCNext2	; (5)
	return				; (6)

FM_PCNext2:
	Nop10				; (7)
	goto		FM_PulseCount	; (17)
	

;;;**********************************************************************
;;; NAME:	FM_ZeroSync()
;;;
;;; DESCR:	Syncronize to an incoming string of zeros.  Returns after
;;;		the first 1 bit is seen.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- just needs to be called in the middle of the zeros
;;;		  somewhere...not picky at all
;;;		- returns at (23)
;;;**********************************************************************
FM_ZeroSync:
	;;
	;; first, wait for a pulse, any pulse...we then get "in sync."
	;; Unlike with DD, the pulse width for SD is darn big, so this loop
	;; will catch it just fine.  Even it it wasn't, we'd eventually
	;; get a good pulse.
	;;

	SkipIfHigh	WDATA		; (0)loop here waiting for a pulse
	GotoPrevious			; (1)

	;; at this point, we're "in" a pulse, and we assume that we
	;; caught the pulse at 0, though it can actually be caught at
	;; 1 or 2 as well.

	;; now that we have a pulse, continue monitoring pulses until
	;; we get one that comes at 20 instead of 40.  At that point,
	;; we have received the first bit of the data mark.

	Nop				; (2)
	
FM_ZSLoop:
	Nop16				; (3)

	WriteSample3R			; (19) returns embedded in macro

	Nop14				; (25)

	;; this will keep us in sync

	WriteSample3	FM_ZSLoop	; (39)

	;; if we fall through here, we somehow missed the next clock
	;; pulse, which is bad...so just return

	return
	
;;;*******************************************************************
;;; NAME:	FM_Write()
;;;
;;; DESCR:	Monitor the WDATA line for incoming FM data.
;;;
;;; ARGS:	WORK0 is the bit count
;;;		WORK1 is the number of bytes to load
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- assumes that we have just seen the last bit, which
;;;		  means that we either "saw it" for a one, or didn't
;;;		  see it, indicating a zero.
;;;		- this routine waits for the clock pulse to stay
;;;		  syncronized.
;;;		- call this at (-3) so that clock pulse monitoring can
;;;		  start at (-1)
;;;		- bytes are written out (if necessary) after the clock
;;;		  pulse is seen.
;;;		- returns at 31 (next inst at 33)
;;;		- BIT NOTE - W is not used in this routine (an important
;;;			fact that some callers use)
;;;*******************************************************************

#define bitcount	WORK0
#define bytecount	WORK1
	
FM_Write:
	WriteSample3	FM_W2		; (-1) get us in sync with clock pulse
	
	;; (NOTE - if we "fall" down here, then we're in trouble)
	;; should probably (somehow) get out of write-mode in this case
	;; reseting memory based upon where WORK1 is

FM_W2:	;; at this point, we have a clock pulse, so we're in sync

	;; we're committed to reading a bit at this point, so
	;; do an early check of the bitcount and bytecount so we can
	;; save a few cycles later upon the time we need to return

	decfsz		bitcount	; (3) check for end of byte
	goto		FM_WNoByteEnd	; (4)

	bsf		bitcount,3	; (5) go ahead and reset bit count
	
	decfsz		bytecount	; (6) check for last byte
	goto		FM_WNoWriteEnd	; (7)

	;; at this point we have an end-of-byte and end-of-write
	;; condition.  Z is used to communicate this condition.
	;; But since decfsz doesn't set it, do it here.
	;; (nothing in this routine changes Z)

	bsf		STATUS,Z	; (8) set Z to indicate end of write
	goto		FM_WCheckBit	; (9) 

FM_WNoByteEnd:
	;; bitcount didn't roll-over (therefore, neither did bytecount)
	;; since decfsz doesn't set Z, it is cleared below
	
	Nop3				; (6)
	
FM_WNoWriteEnd:
	;; bitcount rolled-over but bytecount didn't
	;; since decfsz doesn't set Z, it needs to be clearly explicitely

	bcf		STATUS,Z	; (9)
	Nop1				; (10)

FM_WCheckBit:	

	Nop6				; (11)

	bsf		THISBIT		; (17) default is a one
	ShiftIn		MEMDATA		; (18) get it shifted in

	WriteSample	FM_WOneBit	; (19) a short sample here...

	bcf		LASTBIT		; (23) flip the bit to a zero instead
	Nop1				; (24)

FM_WSaveBit:	
	
	skpz				; (25) Z set if last bit of last byte
	goto		FM_WNextBit	; (26) go on to next bit
	
	MemWriteByte			; (27) write the byte
	MemLowInc			; (29) increment memory

	return				; (31)

FM_WNextBit:
	btfss		bitcount,3	; (28) need to check to see if we have a bit roll-over
	goto		FM_WNextLoop	; (29) nope, just go get next bit

	;; had a byte roll-over at this point
	
	MemWriteByte			; (30) write the byte
	MemLowInc			; (32) increment memory

	;; now we continue to the next bit
	
	Nop3				; (34)
	goto		FM_Write	; (37) go on to next bit

FM_WNextLoop:	
	Nop6				; (31)
	goto		FM_Write	; (37) go on to next bit

FM_WOneBit:
	goto		FM_WSaveBit	; (23) bit defaults to one
	

;;; =======================================================================
;;; =======================================================================

	END
